/** \file     tdd_min.c
 *  \brief    Minimization by restricting to prime implicants
 *  \author   Takahisa Toda
 *  \see      T.Toda, Dualization of Boolean functions Using Ternary Decision Diagrams, in Proc. of 13th International Symposium on Artificial Intelligence and Mathematics (ISAIM2014), Florida, USA(2014).
 */


#include "tdd_min.h"
#include "my_hash.h"

static ztddp min_d2z_rec(dtddp f, my_hash *h);

static uintmax_t recdepth = 0; //!< recursion depth


#ifdef SIZE_LOG
static FILE       *sizlog = NULL;
static uintmax_t maxsize = 0; //!< the maximum size of intermediate ZTDDs
#endif /*SIZE_LOG*/


/** \brief    Construct the ZTDD that represents all prime implicants for the Boolean function represented by a DTDD.
 *  \param    f   DTDD
 *  \return   The ZTDD contructed.
 *  \note     To log output ZTDD sizes, define the macro SIZE_LOG in Makefile. The result is outputed as a file, where the 1st column consists of input sizes and the 2nd column output sizes.
 */
ztddp min_d2z(dtddp f)
{
  my_hash *h = ht_create(0);
  ENSURE_TRUE_MSG(h != NULL, "hash table creation failed");

#ifdef SIZE_LOG
  char str[BUFSIZ];
  strftime(str, BUFSIZ, "tdd_min%Y%m%d%H%M.log", execdate); 
  sizlog = fopen(str, "w");
  ENSURE_TRUE_MSG(sizlog != NULL, "file open failed\n");
  maxsize = 0;
#endif /*SIZE_LOG*/

  assert(recdepth == 0);
  ztddp r = min_d2z_rec(f, h);
  ENSURE_TRUE(r != TDD_NULL);
  assert(recdepth == 0);

#ifdef SIZE_LOG
  fclose(sizlog);
  printf("max|ztdd|\t%ju\n", maxsize);
#endif /*SIZE_LOG*/

  ht_destroy(h);
  return r;
}

static ztddp min_d2z_rec(dtddp f, my_hash *h)
{
  if(f == dtdd_top()) return ztdd_top();
  if(f == dtdd_bot()) return ztdd_bot();

  ztddp r;
  if(ht_search((uintptr_t)f, (uintptr_t*)&r, h)) return r;

  INC_RECDEPTH(recdepth);
  ztddp r0 = min_d2z_rec(dtdd_zero(f), h);  
  ztddp r1 = ztdd_diff(min_d2z_rec(dtdd_neg(f), h), r0);
  ENSURE_TRUE_MSG(r1 != TDD_NULL, "ZTDD operation failed");
  ztddp r2 = ztdd_diff(min_d2z_rec(dtdd_pos(f), h), r0);
  ENSURE_TRUE_MSG(r2 != TDD_NULL, "ZTDD operation failed");
  DEC_RECDEPTH(recdepth);

  r = ztdd_node(dtdd_itemval(f), r0, r1, r2);
  ENSURE_TRUE_MSG(r != TDD_NULL, "ZTDD operation failed");

  ht_insert((uintptr_t)f, (uintptr_t)r, h);
  
#ifdef SIZE_LOG
  const uintmax_t insize  = dtdd_size(f);
  const uintmax_t outsize = ztdd_size(r);
  fprintf(sizlog,"%ju\t%ju\n", insize, outsize);
  if(maxsize < outsize) maxsize = outsize;
#endif /*SIZE_LOG*/

  return r;
}

