/** \file     tdd_io_dimacs.c
 *  \brief    Implementation of IO functions for data files in DIMACS cnf format
 *  \see      T.Toda, Dualization of Boolean functions Using Ternary Decision Diagrams, in Proc. of 13th International Symposium on Artificial Intelligence and Mathematics (ISAIM2014), Florida, USA(2014).
 */
#include <string.h>
#include <time.h>

#include "my_io_dimacs.h"
#include "tdd_io_dimacs.h"
#include "rng.h"
#include "ssort2.h"

static uintmax_t recdepth = 0; //!< recursion depth

static ztddp            ztddconst_rec   (itemval *x[], uintmax_t n, uintmax_t d);
static inline uintmax_t max_itemval     (uintmax_t a, uintmax_t b);
static inline uintmax_t bsearch_itemval (itemval *x[], uintmax_t left, uintmax_t right, itemval v, uintmax_t d);

static inline uintmax_t max_itemval(uintmax_t a, uintmax_t b)
{
  return ((a)<=(b) ? (b) : (a));
}


/** \brief    Build a ZTDD from a DIMACS cnf datafile.
 *  \param    n     The number of rows
 *  \param    k     The total number of entries in a data file
 *  \param    in    Pointer to a dafa file
 *  \return   The computed ZTDD
 */
ztddp ztdd_from_dimacs(uintmax_t n, uintmax_t k, FILE *in)
{
  assert(in != NULL);
  struct setfam *sf = create_setfam_from_dimacs(n, k, in);
  ENSURE_TRUE(sf != NULL);
  format_eachrow(sf->elem, n);
  ssort2main(sf->elem, n);

  assert(recdepth == 0);
  ztddp f = ztddconst_rec(sf->elem, n, 0);
  ENSURE_TRUE(f != TDD_NULL);
  assert(recdepth == 0);

  destroy_setfam(sf);
  return f;
}


/** \brief    Find the first position of an entry with d-th value not equal to v (by a binary search).
 *  \param    x       Double pointer to an item value
 *  \param    left    The leftend position
 *  \param    right   The rightend position
 *  \param    v       Value
 *  \param    d       Depth
 *  \return   The found position
 *  \note     Assume that (v ==) x[left][d] <= x[left+1][d] <= ... <= x[right][d] (!= v)
 */
static inline uintmax_t bsearch_itemval(itemval *x[], uintmax_t left, uintmax_t right, itemval v, uintmax_t d)
{
  assert(x != NULL);
  while(1) {
    assert(x[left][d] == v); assert(x[right][d] != v); assert(left < right);
    if(right - left <= 1) return right;
    uintmax_t mid = (left + right) / 2;
    if(x[mid][d] == v)  left  = mid;
    else                right = mid;
  }
}

/** \brief    Build a ZTDD from a DIMACS CNF datafile.
 *  \param    x     x[i] corresponds to the i-th row in a data file
 *  \param    n     The number of rows
 *  \param    d     The position of rows to be examined.
 *  \return   The computed ZTDD
 */
static ztddp ztddconst_rec(itemval *x[], uintmax_t n, uintmax_t d)
{ 
  assert(x != NULL);
  uintmax_t i = 0; uintmax_t j = 0;
  if (n == 0)     return ztdd_bot();
  itemval v = x[0][d];
  if (v == INFTY) return ztdd_top();
  if(x[n-1][d] == v) {
    if (v < 0)      {i = n; j = n;}
    else if (v > 0) {i = 0; j = n;}
    else            assert(0);
  } else {
    if(v > 0) {
      i = 0;
      for(j = 1; j < n && x[j][d] == v; j *= 2) ;
      if(j < n) j = bsearch_itemval(x, j/2,   j, v, d);
      else      j = bsearch_itemval(x, j/2, n-1, v, d);
    } else {
      for(i = 1; i < n && x[i][d] == v; i *= 2) ;
      if(i < n) i = bsearch_itemval(x, i/2,   i, v, d);
      else      i = bsearch_itemval(x, i/2, n-1, v, d);

      uintmax_t w = -v; 
      if(x[i][d] != w)          {j = i;} 
      else if (x[n-1][d] == w)  {j = n;} 
      else {
  for(j = i; j < n && x[j][d] == w; j *= 2) ;
  if(j < n) j = bsearch_itemval(x, max_itemval(i,j/2),   j, w, d);
  else      j = bsearch_itemval(x, max_itemval(i,j/2), n-1, w, d);
      }
    }
  }

  INC_RECDEPTH(recdepth);
  ztddp r = ztdd_node(abs(v), 
                      ztddconst_rec(x+j, n-j, d),
                      ztddconst_rec(x,   i,   d+1),
                      ztddconst_rec(x+i, j-i, d+1));
  ENSURE_TRUE_MSG(r != TDD_NULL, "ZTDD operation failed");
  DEC_RECDEPTH(recdepth);
  return r;
}


/** \brief    Write a DIMACS cnf datafile from a ZTDD.
 *  \param    m     The maximum value 
 *  \param    f     ZTDD
 *  \param    out   Pointer to a dafa file
 *  \param    s     Output file name
 *  \return   ST_SUCCESS if successful; otherwise, ST_FAILURE.
 */
int ztdd_to_dimacs(itemval m, ztddp f, FILE *out, char *s)
{
  assert(out != NULL);
  
  if(f == ztdd_top()) {
    fprintf(out,"\n");
    return ST_SUCCESS;
  }
  if(f == TDD_NULL || f == ztdd_bot()) return ST_SUCCESS;

  int res = fseek(out, 0L, SEEK_SET);
  ENSURE_SUCCESS(res);

  fprintf(out, "c FILE: %s\n", s); // Insert comment line
  time_t timer;
  timer = time(NULL);
  fprintf(out, "c %s", asctime(localtime(&timer)));
  fprintf(out, "p cnf %d %ju\n", m, ztdd_card(f)); // Insert problem line

  ztddp *br = (ztddp*)malloc(sizeof(ztddp) * (m+1) * 2);
  ENSURE_TRUE_MSG(br != NULL, "memory allocation failed");
  ztddp *bp = br;
  char *sr = (char*)malloc(sizeof(char) * (m+1) * 2);
  ENSURE_TRUE_MSG(sr != NULL, "memory allocation failed");
  char *sp = sr;
  itemval *hi = (itemval*)malloc(sizeof(itemval) * (m+1));
  ENSURE_TRUE_MSG(hi != NULL, "memory allocation failed");
  itemval *hp = hi;

#ifdef MISC_LOG
  printf("ztdd_to_dimacs\t%zu\tbyte\n", malloc_usable_size(br) + malloc_usable_size(sr) + malloc_usable_size(hi));
#endif /*MISC_LOG*/

  while(1) {
    while(f != ztdd_top()) {
      ztddp   no = ztdd_zero(f);
      ztddp   ne = ztdd_neg(f);
      ztddp   po = ztdd_pos(f);
      itemval fv = ztdd_itemval(f);
      if(ne == ztdd_bot()) {
  assert(po != ztdd_bot());
  fprintf(out, "%d ", fv);
  assert(hp < hi + m + 1);
  *hp = fv; hp++;
  if(no != ztdd_bot()) {assert(bp < br + (m+1) * 2); *bp = f; bp++; *sp = 0; sp++;}
  f = po;  
      } else {
  fprintf(out, "-%d ", fv);
  assert(hp < hi + m + 1);
  *hp = -1*fv; hp++;
  if(po != ztdd_bot()) {assert(bp < br + (m+1) * 2); *bp = f; bp++; *sp = 1; sp++;}
  if(no != ztdd_bot()) {assert(bp < br + (m+1) * 2); *bp = f; bp++; *sp = 0; sp++;}
  f = ne;
      }
    }
    fprintf(out, "0 \n");
    if(bp <= br) break;
    bp--; sp--;
    itemval bv = ztdd_itemval(*bp);
    for(hp = hi; abs(*hp) < bv; hp++) fprintf(out, "%d ", *hp);
    if(*sp == 1) {
      f = ztdd_pos(*bp);
      fprintf(out, "%d ", bv);
      assert(hp < hi + m + 1);
      *hp = bv; hp++;   
    }
    else if (*sp == 0) {f = ztdd_zero(*bp);}
    else assert(0);
  }

  free(br); free(sr); free(hi);
  return ST_SUCCESS;
}

