/** \file   tdd_interface.h
 *  \brief  Interface for functions that access TDD packages.
 *  \note   Define your favorite package if you like.
 *  \author Takahisa Toda
 */
#ifndef TDD_INTERFACE_H
#define TDD_INTERFACE_H

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <limits.h>

#include "my_def.h"

#ifdef SELECT_MYTDD
/*---------------------------------------------------------------------------*/
/* MY TDD PACKAGE                                                            */
/*---------------------------------------------------------------------------*/
#include "my_tdd.h"

#define TDD_PACKAGE     "My TDD Package"
#define TDD_NULL        MYTDD_NULL
#define TDD_MAXITEMVAL  MYTDD_MAXITEMVAL //!< maximum item value (or Boolean variable index)

typedef struct mytdd *ztddp; //!< pointer to a ZTDD node
typedef struct mytdd *dtddp; //!< pointer to a DTDD node

/* Common operations */
static inline int tdd_init(uintmax_t cachesiz, uintmax_t flsiz, uintmax_t utsiz, uintmax_t max_utsiz)
{
  int res = mytdd_ut_create(cachesiz, flsiz, utsiz, max_utsiz);
  return res;
}

static inline int tdd_quit(void)
{
  return mytdd_ut_destroy();
}

static inline void tdd_printstat(void)
{
  mytdd_printstat();
}

/* Operations for ZTDDs */
static inline uintmax_t ztdd_size(ztddp f)
{
  return mytdd_size(f); 
}

static inline uintmax_t ztdd_uncmpsize(ztddp f)
{
  return mytdd_uncmpsize(f);
}

static inline uintmax_t ztdd_card(ztddp f)
{
  return mytdd_card(f);
}

static inline ztddp ztdd_bot(void)
{
  return mytdd_bot;
}

static inline ztddp ztdd_top(void)
{
  return mytdd_top;
}

static inline bool ztdd_isconst(ztddp f)
{
  return (f==ztdd_bot() || f==ztdd_top());
}

static inline itemval ztdd_itemval(ztddp f)
{
  return mytdd_itemval(f);
}

static inline ztddp ztdd_pos(ztddp f)
{
  assert(TDD_NULL != f);
  assert(!ztdd_isconst(f));
  return f->po;
}

static inline ztddp ztdd_neg(ztddp f)
{
  assert(TDD_NULL != f);
  assert(!ztdd_isconst(f));
  return f->ne;
}


static inline ztddp ztdd_zero(ztddp f)
{
  assert(TDD_NULL != f);
  assert(!ztdd_isconst(f));
  return f->ze;
}


static inline ztddp ztdd_diff(ztddp f, ztddp g)
{
  return myztdd_diff(f, g);
}

static inline ztddp ztdd_node(itemval v, ztddp z, ztddp n, ztddp p)
{
  return myztdd_node(v, z, n, p);
}

static inline int ztdd_to_dot(itemval v, ztddp f, FILE *out)
{
  return mytdd_to_dot(v, f, out);
}

/* Operations for DTDDs */
static inline uintmax_t dtdd_size(dtddp f)
{
  return mytdd_size(f);
}

static inline uintmax_t dtdd_card(itemval v, dtddp f)
{
  return mydtdd_card(v, f);
}

static inline uintmax_t dtdd_uncmpsize(itemval v, dtddp f)
{
  return mydtdd_uncmpsize(v, f);
}

static inline dtddp dtdd_bot(void)
{
  return mytdd_bot;
}

static inline dtddp dtdd_top(void)
{
  return mytdd_top;
}

static inline bool dtdd_isconst(dtddp f)
{
  return (f==dtdd_bot() || f==dtdd_top());
}

static inline itemval dtdd_itemval(dtddp f)
{
  return mytdd_itemval(f);
}

static inline dtddp dtdd_pos(dtddp f)
{
  assert(TDD_NULL != f);
  assert(!dtdd_isconst(f));
  return f->po;
}

static inline dtddp dtdd_neg(dtddp f)
{
  assert(TDD_NULL != f);
  assert(!dtdd_isconst(f));
  return f->ne;
}


static inline dtddp dtdd_zero(dtddp f)
{
  assert(TDD_NULL != f);
  assert(!dtdd_isconst(f));
  return f->ze;
}

static inline dtddp dtdd_intersect(dtddp f, dtddp g)
{
  return mydtdd_intersect(f, g);
}

static inline dtddp dtdd_node(itemval v, dtddp z, dtddp n, dtddp p)
{
  return mydtdd_node(v, z, n, p);
}

static inline int dtdd_to_dot(itemval v, dtddp f, FILE *out)
{
  return mytdd_to_dot(v, f, out);
}

#endif

#endif /*TDD_INTERFACE_H*/
