/** \file     tdd_gen.c
 *  \brief    Implicant generation
 *  \author   Takahisa Toda
 *  \see      T.Toda, Dualization of Boolean functions Using Ternary Decision Diagrams, in Proc. of 13th International Symposium on Artificial Intelligence and Mathematics (ISAIM2014), Florida, USA(2014).
 */

#include "tdd_gen.h"
#include "my_hash.h"

static dtddp gen_z2d_rec(ztddp f, my_hash *h);

static uintmax_t recdepth = 0; //!< recursion depth

#ifdef SIZE_LOG
static FILE       *sizlog = NULL;
static uintmax_t  maxsize = 0; //!< the maximum size of intermediate DTDDs
#endif /*SIZE_LOG*/


/** \brief    Construct the DTDD for all implicants from a ZTDD, where a ZTDD is considered as a representation of CNFs.
 *  \param    f     ZTDD
 *  \return   The DTDD contructed.
 *  \note     To log output DTDD sizes, define the macro SIZE_LOG in Makefile. The result is outputed as a file, where the 1st column consists of input sizes and the 2nd column output sizes.
 */
dtddp gen_z2d(ztddp f)
{
  my_hash *h = ht_create(0);
  ENSURE_TRUE_MSG(h != NULL, "hash table creation failed");

#ifdef SIZE_LOG
  char str[BUFSIZ];
  strftime(str, BUFSIZ, "tdd_gen%Y%m%d%H%M.log", execdate);
  sizlog = fopen(str, "w");
  ENSURE_TRUE_MSG(sizlog != NULL, "file open failed\n");
  maxsize = 0;
#endif /*SIZE_LOG*/

  assert(recdepth == 0);
  dtddp r = gen_z2d_rec(f, h);
  ENSURE_TRUE(r != TDD_NULL);
  assert(recdepth == 0);

#ifdef SIZE_LOG
  fclose(sizlog);
  printf("max|dtdd|\t%ju\n", maxsize);
#endif /*SIZE_LOG*/

  ht_destroy(h);
  return r;
}

static dtddp gen_z2d_rec(ztddp f, my_hash *h)
{
  if(f == ztdd_top()) return dtdd_bot();
  if(f == ztdd_bot()) return dtdd_top();

  dtddp r;
  if(ht_search((uintptr_t)f, (uintptr_t*)&r, h)) return r;

  dtddp t0 = gen_z2d_rec(ztdd_zero(f), h);  
  dtddp t1 = gen_z2d_rec(ztdd_neg(f),  h);
  dtddp t2 = gen_z2d_rec(ztdd_pos(f),  h);

  INC_RECDEPTH(recdepth);
  dtddp r2 = dtdd_intersect(t0, t1); 
  ENSURE_TRUE_MSG(r2 != TDD_NULL, "DTDD operation failed");
  dtddp r1 = dtdd_intersect(t0, t2); 
  ENSURE_TRUE_MSG(r1 != TDD_NULL, "DTDD operation failed");
  dtddp r0 = dtdd_intersect(r2, t2); 
  ENSURE_TRUE_MSG(r0 != TDD_NULL, "DTDD operation failed");
  DEC_RECDEPTH(recdepth);

  r = dtdd_node(ztdd_itemval(f), r0, r1, r2);
  ENSURE_TRUE_MSG(r != TDD_NULL,  "DTDD operation failed");

  ht_insert((uintptr_t)f, (uintptr_t)r, h);
  
#ifdef SIZE_LOG
  const uintmax_t insize  = ztdd_size(f);
  const uintmax_t outsize = dtdd_size(r);
  fprintf(sizlog,"%ju\t%ju\n", insize, outsize);
  if(maxsize < outsize) maxsize = outsize;
#endif /*SIZE_LOG*/

  return r;
}

