/** \file   ssort2.c
 *  \brief  Implementation of an algorithm for sorting sets of non-zero signed integers.
 *  \note   
 *  - We modified the multikey quicksort by Bentley and Sedgewick: the original source code can be obtained from http://www.cs.princeton.edu/~rs/strings .
 *  - Integers must be less than INFTY.
 *  - The order of signed integers i,j is defined to be i <= j if abs(i) < abs(j) or abs(i) = abs(j) and j > 0.
 */


#include <limits.h>
#include <stdlib.h>
#include <assert.h>

#include "ssort2.h"
#include "rng.h"

#ifdef i2c
#error "The macro i2c is already defined."
#else
#define i2c(i) x[i][depth]
#endif /*i2c*/

static inline uintmax_t min(uintmax_t a, uintmax_t b)
{
  return ((a)<=(b) ? (a) : (b));
}

static inline void swap(uintmax_t a, uintmax_t b, itemval *x[])
{
  itemval *t=x[a];
  x[a]=x[b]; x[b]=t;
}

static inline void vecswap(uintmax_t i, uintmax_t j, uintmax_t n, itemval *x[])
{   while (n-- > 0) {
        swap(i, j, x);
        i++;
        j++;
    }
}

/** \brief    Compare two signed integers based on the order < defined in the top of this file.
 *  \param    a   Signed integer
 *  \param    b   Signed integer
 *  \return   0 if equal; -1 if a < b ;1 if a > b.
 */
static inline itemval comp_sint(itemval a, itemval b)
{
  if(a == b) return 0;
  else if (abs(a) == abs(b)) {
    if(a > 0) {
      return 1;
    } else {
      return -1;
    }
  } else if(abs(a) < abs(b)) {
    return -1;
  } else {
    return 1;
  }
}

static uintmax_t recdepth = 0; //!< recursion depth

static void ssort2(itemval *x[], uintmax_t n, uintmax_t depth)
{   
  uintmax_t a, b, c, d, m;
  itemval   r, v;
  if (n <= 1) return;
  INC_RECDEPTH(recdepth);
  a = rng_rand(n);
  swap(0, a, x);
  v     = i2c(0);
  a = b = 1;
  c = d = n-1;
  for (;;) {
    while (b <= c && (r = comp_sint(i2c(b), v)) <= 0) {
      if (r == 0) { swap(a, b, x); a++;}
      b++;
    }
    while (b <= c && (r = comp_sint(i2c(c), v)) >= 0) {
      if (r == 0) { swap(c, d, x); d--;}
      c--;
    }
    if (b > c) break;
    swap(b, c, x);
    b++;
    c--;
  }
  m = min(a, b-a);     vecswap(0, b-m, m, x);
  m = min(d-c, n-d-1); vecswap(b, n-m, m, x);
  m = b-a; ssort2(x, m, depth);
  assert(i2c(m) == v);
  if (i2c(m) != INFTY) {
    ssort2(x + m, a + n-d-1, depth+1);
    m = d-c; ssort2(x + n - m, m, depth);
  } else {
    assert(d-c == 0);
  }
  DEC_RECDEPTH(recdepth);
}

void ssort2main(itemval *x[], uintmax_t n)
{
  assert(recdepth == 0);
  assert(x != NULL);
  ssort2(x, n, 0);
  assert(recdepth == 0);
}

