/** \file     my_tdd.h
 *  \brief    TDD package
 *  \author   Takahisa Toda
 */
#ifndef MY_TDD_H
#define MY_TDD_H

#include <stdlib.h>
#include <limits.h>
#include "my_def.h"

#define MYTDD_NULL                   NULL

#define MYTDD_MAXITEMVAL             (INFTY-1)           //!< the maximum item value (or Boolean variable index)


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

/** \brief  ternary decidion diagram node*/
struct mytdd {
  itemval       v;        //!< item value, where its sign is used to decide if this node is checked in traversing a TDD.
  uintptr_t     aux;      //!< an auxiliary field (of pointer size), which is introduced in order to facilitate implementation.
  struct mytdd  *ze;      //!< zero arc
  struct mytdd  *ne;      //!< negative arc
  struct mytdd  *po;      //!< positive arc
  struct mytdd  *nx;      //!< a link that connect TDD nodes in the same bucket for resolving collisions.
};


extern struct mytdd *mytdd_top; //!< top terminal node
extern struct mytdd *mytdd_bot; //!< bottom terminal node

/*---------------------------------------------------------------------------*/
/* Function prototypes                                                       */
/*---------------------------------------------------------------------------*/

// common operations
extern int            mytdd_ut_create     (uintmax_t cachesiz, uintmax_t flsiz, uintmax_t utsiz, uintmax_t max);
extern int            mytdd_ut_destroy    (void);
extern uintmax_t      mytdd_size          (struct mytdd *f);
extern int            mytdd_to_dot        (itemval maxval, struct mytdd *f, FILE *out);
extern uintmax_t      mytdd_uncmpsize     (struct mytdd *f); //!< for ZTDD or QTDD only 
extern uintmax_t      mytdd_card          (struct mytdd *f); //!< for ZTDD or QTDD only

static inline itemval mytdd_itemval       (struct mytdd *f)
{
  assert(f != NULL);
  return abs(f->v);
}

static inline void    mytdd_setitemval    (itemval v, struct mytdd *f)
{
  assert(f != NULL);
  f->v = abs(v);
}

extern struct mytdd   *mydtdd_node        (itemval v, struct mytdd *z, struct mytdd *n, struct mytdd *p);
extern struct mytdd   *mydtdd_intersect   (struct mytdd *f, struct mytdd *g);
extern struct         mytdd *mytdd_d2q    (itemval maxval, struct mytdd *f);

static inline uintmax_t mydtdd_card       (itemval v, struct mytdd *f)
{
  struct mytdd *g = mytdd_d2q(v, f);
  return mytdd_card(g);
}

static inline uintmax_t mydtdd_uncmpsize   (itemval v, struct mytdd *f)
{
  struct mytdd *g = mytdd_d2q(v, f);
  return mytdd_uncmpsize(g);
}

extern struct mytdd   *myztdd_node        (itemval v, struct mytdd *z, struct mytdd *n, struct mytdd *p);
extern struct mytdd   *myztdd_diff        (struct mytdd *f, struct mytdd *g);

extern struct mytdd   *myqtdd_node        (itemval v, struct mytdd *z, struct mytdd *n, struct mytdd *p);

extern void           mytdd_printstat     (void);

#endif /*MY_TDD_H*/
