/** \file     my_tdd.c
 *  \brief    A simple implementation of TDD package
 *  \see      T.Toda, Dualization of Boolean functions Using Ternary Decision Diagrams, in Proc. of 13th International Symposium on Artificial Intelligence and Mathematics (ISAIM2014), Florida, USA(2014).
 */
#include <stdio.h>
#include <stdint.h>
#include <assert.h>
#include <malloc.h>

#include "my_tdd.h"
#include "my_def.h"
#include "my_opcache.h"

#define MYTDD_UT_DEFAULT_BUCKET_COUNT     (UINTMAX_C(1) << 24)//!< the initial number of buckets in uniquetable
#define MYTDD_UT_DEFAULT_MAX_USABLE_SIZE  (UINTMAX_C(1) << 32)//!< the maximum usable memory (uniquetable + operation cache) in byte.
#define MYTDD_UT_MAX_LOADFACTOR           (7)                 //!< the maximum load factor of uniquetable
#define MYTDD_UT_ENLARGEMENT_WIDTH        (2)                 //!< the width of bitshift for enlargement

#define MYTDD_FL_DEFAULT_LEN              (UINTMAX_C(1) << 24)//!< the initial length of a freelist for TDD nodes
#define MYTDD_FL_ENLARGEMENT_WIDTH        (1)                 //!< the width of bitshift for enlargement

#define HASHCONST                         (31415926525)

static inline bool          mytdd_isvisited       (struct mytdd *f);
static inline void          mytdd_setvisited      (struct mytdd *f);
static inline void          mytdd_unsetvisited    (struct mytdd *f);
static void                 mytdd_unsetvisited_rec(struct mytdd *f);

static uintmax_t            mytdd_size_rec        (struct mytdd *f);
static uintmax_t            mytdd_card_rec        (struct mytdd *f);
static uintmax_t            mytdd_uncmpsize_rec   (struct mytdd *f);

static void                 mytdd_setaux          (intptr_t v, struct mytdd *f);

static inline struct mytdd  *mytdd_node           (itemval v, struct mytdd *z, struct mytdd *n, struct mytdd *p);
static inline uintmax_t     mytdd_hash            (itemval v, const struct mytdd *z, const struct mytdd *n, const struct mytdd *p, uintmax_t s);
static inline struct mytdd  *mytdd_alloc          (void);
static inline void          mytdd_reclaim         (struct mytdd *f);

static struct mytdd         *mytdd_d2q_rec        (itemval m, struct mytdd *f);
static struct mytdd         *mytdd_restore_nodes  (itemval n, itemval m, struct mytdd *f);

static void                 mytdd_ut_enlarge      (void);

static struct mytdd         **mytdd_create_arrayoflists (itemval m, struct mytdd *f);
static void                 mytdd_make_arrayoflists     (struct mytdd *f, struct mytdd **a);
static void                 mytdd_destroy_arrayoflists  (struct mytdd **a);

static inline void          mytdd_add_allocated_size    (void *p);
static inline void          mytdd_del_allocated_size    (void *p);

#ifdef MY_DEBUG
#include "my_hash.h"
static bool                 mytdd_isconsistent          (struct mytdd *f);
static bool                 mytdd_isconsistent_rec      (struct mytdd *f, my_hash *h);
#endif /*MY_DEBUG*/

#ifdef UT_LOG
static uintmax_t mytdd_collision_count  = 0;
static uintmax_t mytdd_call_count       = 0;
#endif /*UT_LOG*/

struct mytdd *mytdd_top = NULL;
struct mytdd *mytdd_bot = NULL;

static bool           mytdd_ut_exists                = false;
static struct mytdd **mytdd_ut                       = NULL; //!< a uniquetable for TDD nodes
static uintmax_t      mytdd_ut_bucket_count          = MYTDD_UT_DEFAULT_BUCKET_COUNT;
static uintmax_t      mytdd_ut_node_count            = 0;
static uintmax_t      mytdd_ut_node_count_threshold  = MYTDD_UT_DEFAULT_BUCKET_COUNT * MYTDD_UT_MAX_LOADFACTOR; //!< threshold for deciding enlargement.
static uintmax_t      mytdd_ut_max_usable_size       = MYTDD_UT_DEFAULT_MAX_USABLE_SIZE;
static uintmax_t      mytdd_ut_allocated_size        = 0; //!< dynamically allocated memory size in byte for uniquetable (without cache)
static uintmax_t      mytdd_opcache_allocated_size   = 0; //!< dynamically allocated memory size in byte for operation cache

static my_opcache *mytdd_opcache = NULL;
enum {
  MYTDD_SIZE_OP = 0,
  MYTDD_CARD_OP,
  MYTDD_UNCMPSIZE_OP,
  MYDTDD_INT_OP,
  MYZTDD_DIFF_OP,
  MYTDD_D2Q_OP,
};

static uintmax_t    mytdd_fl_len  = MYTDD_FL_DEFAULT_LEN;
static struct mytdd *mytdd_fl     = NULL; //!< a freelist for TDD nodes
static struct mytdd *dummy        = NULL; //!< a linked list of dummy TDD nodes, which are used only for freeing allocated TDD nodes.

static uintmax_t recdepth         = 0;    //!< recursion depth


/** \brief    Create and initialize a uniquetable for TDD nodes.
 *  \param    cachesiz    The initial number of operation cache entries
 *  \param    flsiz       The initial lengh of a freelist for TDD nodes
 *  \param    utsiz       The initial number of buckets of uniquetable 
 *  \param    max         The maximum usable memory in M byte
 *  \return   ST_SUCCESS if successful; otherwise, ST_FAILURE.
 *  \note
 *  - Nothing is done if a ut already exists.
 *  - Before executing other functions, execute this function once.
 *  - Set default values if cachesiz, flsiz, utsiz, and max equal 0.
 */
int mytdd_ut_create(uintmax_t cachesiz, uintmax_t flsiz, uintmax_t utsiz, uintmax_t max)
{
#if MYTDD_MAXITEMVAL >= INFTY
#error "MYTDD_MAXITEMVAL must be less than INFTY"
#endif

  if(mytdd_ut_exists) {
    ENSURE_TRUE_WARN(false, "TDD ut already exists");
    return ST_FAILURE;
  }

  recdepth                      = 0;

  mytdd_fl_len                  = (flsiz == 0? MYTDD_FL_DEFAULT_LEN: flsiz);
  mytdd_fl                      = NULL;
  dummy                         = NULL;

  mytdd_ut_bucket_count         = (utsiz == 0? MYTDD_UT_DEFAULT_BUCKET_COUNT: utsiz);
  mytdd_ut_node_count           = 0;
  mytdd_ut_node_count_threshold = mytdd_ut_bucket_count * MYTDD_UT_MAX_LOADFACTOR;
  mytdd_ut_max_usable_size      = (max == 0? MYTDD_UT_DEFAULT_MAX_USABLE_SIZE: max);
  mytdd_ut_allocated_size       = 0;

#ifdef UT_LOG
  mytdd_collision_count         = 0;
  mytdd_call_count              = 0;
  printf("tdd_node\t%zu\tbyte\n", sizeof(struct mytdd));
  printf("ut_max_usable\t%ju\tbyte\n", mytdd_ut_max_usable_size);
  printf("\n");
#endif /*UT_LOG*/

  mytdd_ut = (struct mytdd**) malloc(sizeof(struct mytdd*) * mytdd_ut_bucket_count);
  ENSURE_TRUE_MSG(mytdd_ut != NULL, "memory allocation failed");
  mytdd_add_allocated_size(mytdd_ut);
  for (uintmax_t i = 0; i < mytdd_ut_bucket_count; i++) mytdd_ut[i] = NULL;

  mytdd_top     = mytdd_alloc();
  ENSURE_TRUE_MSG(mytdd_top != NULL, "memory allocation failed");
  mytdd_top->ze = NULL;
  mytdd_top->ne = NULL;
  mytdd_top->po = NULL;
  mytdd_top->nx = NULL;
  mytdd_setitemval(INFTY, mytdd_top);
  mytdd_setaux    (0,     mytdd_top);
  mytdd_setvisited(mytdd_top);

  mytdd_bot     = mytdd_alloc();
  ENSURE_TRUE_MSG(mytdd_bot != NULL, "memory allocation failed");
  mytdd_bot->ze = NULL;
  mytdd_bot->ne = NULL;
  mytdd_bot->po = NULL;
  mytdd_bot->nx = NULL;
  mytdd_setitemval(INFTY, mytdd_bot);
  mytdd_setaux    (0,     mytdd_bot);
  mytdd_setvisited(mytdd_bot);

  mytdd_opcache = opcache_create(cachesiz);
  ENSURE_TRUE_MSG(mytdd_opcache != NULL, "opcache creation failed");
  mytdd_opcache_allocated_size = opcache_allocsize(mytdd_opcache);

  mytdd_ut_exists = true;
  return ST_SUCCESS;
}

/** \brief    Free the memory area used by a uniquetable.
 *  \return   ST_SUCCESS if successful; otherwise, ST_FAILURE.
 *  \note     Nothing is done if a uniquetable does not exist.
 */
int mytdd_ut_destroy(void)
{
  if(!mytdd_ut_exists) {
    ENSURE_TRUE_WARN(false, "TDD ut no longer exists");
    return ST_FAILURE;
  }

  for(struct mytdd *e = dummy; e != NULL;) {
    struct mytdd *t = e->nx;
    mytdd_del_allocated_size(e);
    free(e);
    e = t;
  }
  mytdd_del_allocated_size(mytdd_ut);
  free(mytdd_ut);  mytdd_ut   = NULL;

  mytdd_opcache_allocated_size -= opcache_allocsize(mytdd_opcache);
  opcache_destroy(mytdd_opcache);
  mytdd_opcache   = NULL;

  ENSURE_TRUE_WARN(mytdd_ut_allocated_size      == 0, "memory leak was detected");
  ENSURE_TRUE_WARN(mytdd_opcache_allocated_size == 0, "memory leak was detected");

  mytdd_ut_exists = false;
  return ST_SUCCESS;
}


/** \brief    Add a dynamically allocated memory size.
 *  \param    p     Pointer to a dynamically allocated memory block.
 *  \note     Call this function just after a memory block is newly allocated!
 *  \see      mytdd_del_allocated_size
 */
static inline void  mytdd_add_allocated_size    (void *p)
{
  uintmax_t n = malloc_usable_size(p);
  assert(mytdd_ut_allocated_size <= UINTMAX_MAX -n);
  mytdd_ut_allocated_size += n;
  ENSURE_TRUE_MSG(mytdd_ut_allocated_size + mytdd_opcache_allocated_size <= mytdd_ut_max_usable_size, "maximum usable memory exceeded");
}

/** \brief    Delete the size of a memory that is about to be freed.   
 *  \param    p   Pointer to a memory block
 *  \note     Call this function just before a memory block is freed!
 *  \see      mytdd_add_allocated_size
 */
static inline void  mytdd_del_allocated_size     (void *p)
{
  uintmax_t n = malloc_usable_size(p);
  assert(mytdd_ut_allocated_size >= n);
  mytdd_ut_allocated_size -= n;
}

/** \brief    Compute the load factor of uniquetable.
 *  \return   The computed value
 */
double mytdd_ut_loadfactor(void)
{
  return (double)mytdd_ut_node_count/mytdd_ut_bucket_count;
}

/** \brief    Calculate a hash value.
 *  \param    k0    Key0
 *  \param    k1    Key1
 *  \param    k2    Key2
 *  \param    k3    Key3
 *  \param    n     The maximum hash value plus 1
 *  \return   The calculated hash value.
 */
static inline uintmax_t mytdd_hash(itemval k0, const struct mytdd *k1, const struct mytdd *k2, const struct mytdd *k3, uintmax_t n)
{
  assert(n > 0);
  uintptr_t hash = 0;
  hash = hash * HASHCONST + (uintptr_t)k0;
  hash = hash * HASHCONST + ((uintptr_t)k1 >> 3);
  hash = hash * HASHCONST + ((uintptr_t)k2 >> 3);
  hash = hash * HASHCONST + ((uintptr_t)k3 >> 3);
  return (uintmax_t)(hash%n);
}

/** \brief    Obtain an unsed TDD node from a freelist. If a freelist is empty, allocate TDD nodes.
 *  \return   TDD node
 *  \note     TDD nodes must be obtained by this function. 
 */
static inline struct mytdd *mytdd_alloc(void)
{
  if(mytdd_fl == NULL) {
    ENSURE_TRUE_MSG(mytdd_fl_len < (UINTMAX_MAX >> MYTDD_FL_ENLARGEMENT_WIDTH), "integer overflow");
    mytdd_fl_len = (mytdd_fl_len << MYTDD_FL_ENLARGEMENT_WIDTH);
    struct mytdd *s = (struct mytdd *)malloc(sizeof(struct mytdd) * mytdd_fl_len); 
    ENSURE_TRUE_MSG(s != NULL, "memory allocation failed");
    mytdd_add_allocated_size(s);
    s->nx = dummy;
    dummy = s;
    struct mytdd *t = NULL;
    for(uintmax_t i = 1; i < mytdd_fl_len; i++) {
      (++s)->nx = t;
      t         = s;
    }
    mytdd_fl = s;
  }

  struct mytdd *t = mytdd_fl;
  mytdd_fl  = t->nx;
  t->nx     = NULL;
  return t;
}

/** \brief    Reclaim an unused TDD node to a freelist.
 *  \note     Reclaim TDD nodes if they turn out to be obsolete and never be referenced in future.
 */
static inline void mytdd_reclaim(struct mytdd *f)
{
  assert(f != NULL);
  f->nx    = mytdd_fl;
  mytdd_fl = f;
}

/** \brief    Find a TDD node with speficied fields in a uniquetable.
 *  \param    v     Item value 
 *  \param    z     Pointer to a zero child
 *  \param    n     Pointer to a negative child
 *  \param    p     Pointer to a positive child
 *  \return   TDD node
 *  \note
 *  - If such a TDD node does not exists, it is created and then registered.
 *  - This function can be called only by myztdd_node, mydtdd_node, and myqtdd_node.
 *  \see      mytdd_ut_enlarge
 */
static inline struct mytdd *mytdd_node(itemval v, struct mytdd *z, struct mytdd *n, struct mytdd *p)
{
#ifdef UT_LOG
  mytdd_call_count++;
#endif /*UT_LOG*/

  const uintmax_t i = mytdd_hash(v, z, n, p, mytdd_ut_bucket_count);
  for(struct mytdd *e = mytdd_ut[i]; e != NULL; e = e->nx) {
#ifdef UT_LOG
      mytdd_collision_count++;
#endif /*UT_LOG*/
    if(mytdd_itemval(e) == v && e->ze == z && e->ne == n && e->po == p) {
      return e;
    }
  }

  struct mytdd *new = mytdd_alloc();
  ENSURE_TRUE_MSG(new != NULL, "memory allocation failed");
  new->ze = z; new->ne = n; new->po = p;
  mytdd_setitemval  (v, new);
  mytdd_setaux      (0, new);
  mytdd_unsetvisited(new);
  new->nx     = mytdd_ut[i];
  mytdd_ut[i] = new;
  mytdd_ut_node_count++;

  if(mytdd_ut_node_count > mytdd_ut_node_count_threshold) mytdd_ut_enlarge();

  return new;
}



/** \brief    Find a DTDD node with speficied fields in a uniquetable.
 *  \param    v     Item value
 *  \param    z     Pointer to a zero child 
 *  \param    n     Pointer to a negative child
 *  \param    p     Pointer to a positive child
 *  \return   DTDD node
 */
struct mytdd *mydtdd_node(itemval v, struct mytdd *z, struct mytdd *n, struct mytdd *p)
{
  assert(z != NULL); assert(n != NULL); assert(p != NULL);
  assert(v < mytdd_itemval(z)); assert(v < mytdd_itemval(n)); assert(v < mytdd_itemval(p));

  if(z == n && n == p)  return z;
  else                  return mytdd_node(v, z, n, p);
}


/** \brief    Find a ZTDD node with speficied fields in a uniquetable.
 *  \param    v     Item value
 *  \param    z     Pointer to a zero child
 *  \param    n     Pointer to a negative child
 *  \param    p     Pointer to a positive child
 *  \return   ZTDD
 */
struct mytdd *myztdd_node(itemval v, struct mytdd *z, struct mytdd *n, struct mytdd *p)
{
  assert(z != NULL); assert(n != NULL); assert(p != NULL);
  assert(v < mytdd_itemval(z)); assert(v < mytdd_itemval(n)); assert(v < mytdd_itemval(p));

  if(n == p && p == mytdd_bot)  return z;
  else                          return mytdd_node(v, z, n, p);
}

/** \brief    Find a QTDD node with speficied fields in a uniquetable.
 *  \param    v     Item value
 *  \param    z     Pointer to a zero child
 *  \param    n     Pointer to a negative child
 *  \param    p     Pointer to a positive child
 *  \return   QTDD
 *  \note     QTDD means a TDD having node sharing rule but not having node elimination rule.
 */
struct mytdd *myqtdd_node(itemval v, struct mytdd *z, struct mytdd *n, struct mytdd *p)
{
  assert(z != NULL); assert(n != NULL); assert(p != NULL);
  assert(v+1 == mytdd_itemval(z) || mytdd_itemval(z) == INFTY);
  assert(v+1 == mytdd_itemval(n) || mytdd_itemval(n) == INFTY);
  assert(v+1 == mytdd_itemval(p) || mytdd_itemval(p) == INFTY);

  return mytdd_node(v, z, n, p);
}

/** \brief    Set a value to the aux field of a TDD node.
 *  \param    v     value of pointer size
 *  \param    f     TDD
 *  \note
 *  - Be careful for overflow because the bitwidth for pointer may be smaller than that for uintmax_t.
 *  - AUX field must not be accessed by other means than this function.
 */
static void mytdd_setaux(intptr_t v, struct mytdd *f)
{
  f->aux = v;
}

/** \brief    Unset visited flags recursively.
 *  \param    f     TDD
 *  \note     For safety, call this function to access visited flag and do not access directly.
 */
static void mytdd_unsetvisited_rec(struct mytdd *f)
{
  assert(f != NULL);
  if(f == mytdd_top || f == mytdd_bot) return;
  if(mytdd_isvisited(f)) {
    INC_RECDEPTH(recdepth);
    mytdd_unsetvisited_rec(f->ze);
    mytdd_unsetvisited_rec(f->ne);
    mytdd_unsetvisited_rec(f->po);
    DEC_RECDEPTH(recdepth);  
    mytdd_unsetvisited(f);
  }
}

/** \brief    Check if a visited flag is set.
 *  \param    f     TDD node
 *  \return   true if a visited flag is set; otherwise, false.
 *  \note     For safety, call this function to access visited flag and do not access directly.
 */
static inline bool mytdd_isvisited(struct mytdd *f)
{
  assert(f != NULL);
  return f->v < 0;
}


/** \brief    Set a visited flag.
 *  \param    f     TDD node
 *  \note
 *  - Do not forget to unset!
 *  - For safety, call this function to access visited flag and do not access directly.
 */
static inline void mytdd_setvisited(struct mytdd *f)
{
  assert(f != NULL);
  if(f->v > 0) f->v *= -1;
}

/** \brief    Unset a visited flag.
 *  \param    f     TDD node
 *  \note     For safety, call this function to access visited flag and do not access directly.
 */
static inline void mytdd_unsetvisited(struct mytdd *f)
{
  assert(f != NULL);
  if(f->v < 0) f->v *= -1;
}


#ifdef MY_DEBUG
/** \brief    Check if all nodes in a TDD are consistent.
 *  \param    f     TDD
 *  \return   true if no problem; otherwise, false.
 */
bool mytdd_isconsistent(struct mytdd *f)
{
  assert(f != NULL);
  my_hash *h = ht_create(0);
  ENSURE_TRUE_MSG(h != NULL, "hash creation failed");
  assert(recdepth == 0);
  bool res = mytdd_isconsistent_rec(f, h);
  assert(recdepth == 0);
  ht_destroy(h);
  return res;
}


static bool mytdd_isconsistent_rec(struct mytdd *f, my_hash *h)
{
  assert(f != NULL); assert(h != NULL);
  if(f == mytdd_top || f == mytdd_bot) {
    ENSURE_TRUE_WARN(mytdd_itemval(f) == INFTY, "the v field is invalid");
    ENSURE_TRUE_WARN(f->ze            == NULL,  "the zero field is invalid");
    ENSURE_TRUE_WARN(f->ne            == NULL,  "the neg field is invalid");
    ENSURE_TRUE_WARN(f->po            == NULL,  "the pos field is invalid");
    ENSURE_TRUE_WARN(f->nx            == NULL,  "the nx field is invalid");
    ENSURE_TRUE_WARN(mytdd_isvisited(f),        "the visited field is invalid");
    return (mytdd_itemval(f) == INFTY) && (f->ze == NULL) && (f->ne == NULL) && (f->po == NULL) && (f->nx == NULL) && mytdd_isvisited(f);
  }

  uintptr_t res;
  if(ht_search((uintptr_t)f, (uintptr_t*)&res, h)) return (bool)res;

  INC_RECDEPTH(recdepth);
  if(!mytdd_isconsistent_rec(f->ze, h)) {DEC_RECDEPTH(recdepth); return false;}
  if(!mytdd_isconsistent_rec(f->ne, h)) {DEC_RECDEPTH(recdepth); return false;}
  if(!mytdd_isconsistent_rec(f->po, h)) {DEC_RECDEPTH(recdepth); return false;}
  DEC_RECDEPTH(recdepth);
  
  ENSURE_TRUE_WARN(0 < mytdd_itemval(f) && mytdd_itemval(f) <= MYTDD_MAXITEMVAL, "the v field is invalid");
  ENSURE_TRUE_WARN(f->ze != NULL, "the zero field is invalid");
  ENSURE_TRUE_WARN(f->ne != NULL, "the neg field is invalid");
  ENSURE_TRUE_WARN(f->po != NULL, "the pos field is invalid");
  ENSURE_TRUE_WARN(!mytdd_isvisited(f), "the visited field is invalid");
  res = (uintptr_t)(0 < mytdd_itemval(f) && mytdd_itemval(f) <= MYTDD_MAXITEMVAL && f->ze != NULL && f->ne != NULL && f->po != NULL && !mytdd_isvisited(f));

  ht_insert((uintptr_t)f, (uintptr_t)res, h);
  return (bool)res;
}
#endif /*MY_DEBUG*/

/** \brief    Convert a DTDD to a QTDD by restoring all nodes that are deleted by the node elimination rule of DTDDs.
 *  \param    m     The maximum item value in f
 *  \param    f     DTDD
 *  \return   The converted QTDD
 */
struct mytdd *mytdd_d2q(itemval m, struct mytdd *f)
{
#ifdef MY_DEBUG
  ENSURE_TRUE(mytdd_isconsistent(f));
#endif /*MY_DEBUG*/
  assert(f != NULL);
  assert(recdepth == 0);
  struct mytdd *r = mytdd_d2q_rec(m, f);
  assert(recdepth == 0);
  r = mytdd_restore_nodes(1, m, r);
#ifdef MY_DEBUG
  ENSURE_TRUE(mytdd_isconsistent(r));
#endif /*MY_DEBUG*/
  return r;
}

/** \brief  Restore hidden nodes above the root of a given TDD so that a new root node has n as item value.
 *  \param  n     A restore point
 *  \param  m     The maximum item value in f
 *  \param  f     TDD
 *  \return The computed TDD
 *  \see    mytdd_d2q
 */
static struct mytdd *mytdd_restore_nodes(itemval n, itemval m, struct mytdd *f)
{
  assert(f != NULL);
  assert(n <= m);
  itemval s;
  if(f == mytdd_top || f == mytdd_bot)  s = m;
  else                                  s = mytdd_itemval(f) - 1;
  assert(s >= n);
  struct mytdd *r = f;
  for(itemval i = s; i >= n; i--) {
    r = myqtdd_node(i, r, r, r);
    ENSURE_TRUE_MSG(r != NULL, "TDD operation failed");
  }
  return r;
}


/** \brief    Restore all hidden nodes below f.
 *  \param    m   The maximum item value in f
 *  \param    f   TDD
 *  \return   The computed TDD
 *  \see      mytdd_d2q
 */
static struct mytdd *mytdd_d2q_rec(itemval m, struct mytdd *f)
{
  assert(f != NULL);
  if(f == mytdd_top || f == mytdd_bot) return f;

  struct mytdd *r;
  if(opcache_search(MYTDD_D2Q_OP, (uintptr_t)m, (uintptr_t)f, (uintptr_t*)&r, mytdd_opcache)) return r;
  
  INC_RECDEPTH(recdepth);
  struct mytdd *r0 = mytdd_d2q_rec(m, f->ze);
  struct mytdd *r1 = mytdd_d2q_rec(m, f->ne);
  struct mytdd *r2 = mytdd_d2q_rec(m, f->po);
  DEC_RECDEPTH(recdepth);
  r0 = mytdd_restore_nodes(mytdd_itemval(f)+1, m, r0);
  r1 = mytdd_restore_nodes(mytdd_itemval(f)+1, m, r1);
  r2 = mytdd_restore_nodes(mytdd_itemval(f)+1, m, r2);

  r = myqtdd_node(mytdd_itemval(f), r0, r1, r2);
  ENSURE_TRUE_MSG(r != NULL, "TDD operation failed");
  opcache_insert(MYTDD_D2Q_OP, (uintptr_t)m, (uintptr_t)f, (uintptr_t)r, mytdd_opcache);
  return r;
}


/** \brief    Compute the number of internal nodes in a TDD.
 *  \param    f     TDD
 *  \return   The computed size
 *  \note     
 *  - This function does not count terminal nodes.
 *  - Make sure that the resulted size can be contained in a variable of pointer size!
 */
uintmax_t mytdd_size(struct mytdd *f)
{
  assert(f != NULL);
#ifdef MY_DEBUG
  ENSURE_TRUE(mytdd_isconsistent(f));
#endif /*MY_DEBUG*/

  uintptr_t s;
  if(opcache_search(MYTDD_SIZE_OP, (uintptr_t)f, 0, (uintptr_t*)&s, mytdd_opcache)) return (uintmax_t)s;

  assert(recdepth == 0);
  s = mytdd_size_rec(f);
  assert(recdepth == 0);

  opcache_insert(MYTDD_SIZE_OP, (uintptr_t)f, 0, (uintptr_t)s, mytdd_opcache);
  mytdd_unsetvisited_rec(f);
  return (uintmax_t)s;
}

static uintmax_t mytdd_size_rec(struct mytdd *f)
{
  assert(f != NULL);
  if(mytdd_isvisited(f)) return 0;

  INC_RECDEPTH(recdepth);
  uintmax_t s0 = mytdd_size_rec(f->ze);
  uintmax_t s1 = mytdd_size_rec(f->ne);
  uintmax_t s2 = mytdd_size_rec(f->po);
  DEC_RECDEPTH(recdepth);
  ENSURE_TRUE_MSG(s0      <= UINTPTR_MAX,       "integer overflow");
  ENSURE_TRUE_MSG(s1      <= UINTPTR_MAX,       "integer overflow");
  ENSURE_TRUE_MSG(s2      <= UINTPTR_MAX,       "integer overflow");
  ENSURE_TRUE_MSG(s0      <= UINTPTR_MAX - s1,  "integer overflow");
  ENSURE_TRUE_MSG(s0 + s1 <  UINTPTR_MAX - s2,  "integer overflow");
  uintptr_t s = s0 + s1 + s2 + 1;

  mytdd_setvisited(f);
  return s;
}

/** \brief    Compute the cardinality of a TDD, that is, the number of paths from the root to top terminal nodes.
 *  \param    f     ZTDD or a non-constant QTDD
 *  \return   The computed number
 *  \note     After execution, each node holds the number of paths from this node to top terminal nodes in the aux field.
 */
uintmax_t mytdd_card(struct mytdd *f)
{
  assert(f != NULL);
#ifdef MY_DEBUG
  ENSURE_TRUE(mytdd_isconsistent(f));
#endif /*MY_DEBUG*/

  assert(recdepth == 0);
  uintmax_t c = mytdd_card_rec(f);
  assert(recdepth == 0);

  mytdd_unsetvisited_rec(f);
  return c;
}

static uintmax_t mytdd_card_rec(struct mytdd *f)
{
  assert(f != NULL);
  if(f == mytdd_top) {mytdd_setaux(1,f); return 1;}
  if(f == mytdd_bot) {mytdd_setaux(0,f); return 0;}

  if(mytdd_isvisited(f)) return (uintmax_t)(f->aux);
 
  INC_RECDEPTH(recdepth);
  uintmax_t c0 = mytdd_card_rec(f->ze);
  uintmax_t c1 = mytdd_card_rec(f->ne);
  uintmax_t c2 = mytdd_card_rec(f->po);
  ENSURE_TRUE_WARN(c0       <= UINTPTR_MAX - c1, "integer overflow");
  ENSURE_TRUE_WARN(c0 + c1  <= UINTPTR_MAX - c2, "integer overflow");
  DEC_RECDEPTH(recdepth);
  uintptr_t c  = c0 + c1 + c2;

  mytdd_setaux((uintptr_t)c, f);
  mytdd_setvisited(f);
  return (uintmax_t)c;
}

/** \brief    Compute the uncompressed size of f, that is, the number of literals with repetition when f is decompressed into a CNF.
 *  \param    f     ZTDD or non-constant QTDD
 *  \return   The computed size
 *  \note
 *  - Input TDD must be a non-constant node and must not be a DTDD.
 *  - Make sure that the resulted size can be contained in a variable of pointer size!
 */
uintmax_t mytdd_uncmpsize(struct mytdd *f)
{
  assert(f != NULL);
#ifdef MY_DEBUG
  ENSURE_TRUE(mytdd_isconsistent(f));
#endif /*MY_DEBUG*/

  mytdd_card(f); // Compute cardinalities and set to AUX fields for later use. 
  assert(recdepth == 0);
  uintmax_t s = mytdd_uncmpsize_rec(f);
  assert(recdepth == 0);
  return s;
}

static uintmax_t mytdd_uncmpsize_rec(struct mytdd *f)
{
  assert(f != NULL);
  if(f == mytdd_top || f == mytdd_bot) return 0;

  uintptr_t s;
  if(opcache_search(MYTDD_UNCMPSIZE_OP, (uintptr_t)f, 0, (uintptr_t*)&s, mytdd_opcache)) return (uintmax_t)s;

  INC_RECDEPTH(recdepth);
  uintmax_t s0 = mytdd_uncmpsize_rec(f->ze);
  uintmax_t s1 = mytdd_uncmpsize_rec(f->ne);
  uintmax_t s2 = mytdd_uncmpsize_rec(f->po);
  DEC_RECDEPTH(recdepth);
  ENSURE_TRUE_WARN(s0       <= UINTPTR_MAX,      "integer overflow");
  ENSURE_TRUE_WARN(s1       <= UINTPTR_MAX,      "integer overflow");
  ENSURE_TRUE_WARN(s2       <= UINTPTR_MAX,      "integer overflow");
  ENSURE_TRUE_WARN(s0       <= UINTPTR_MAX - s1, "integer overflow");
  ENSURE_TRUE_WARN(s0 + s1  <= UINTPTR_MAX - s2, "integer overflow");
  s = s0 + s1 + s2;

  uintmax_t c1 = f->ne->aux;
  uintmax_t c2 = f->po->aux;
  ENSURE_TRUE_WARN(s      <= UINTPTR_MAX - c1, "integer overflow");
  ENSURE_TRUE_WARN(s + c1 <= UINTPTR_MAX - c2, "integer overflow");
  s += (c1 + c2);

  opcache_insert(MYTDD_UNCMPSIZE_OP, (uintptr_t)f, 0, (uintptr_t)s, mytdd_opcache);
  return (uintmax_t)s;
}

/** \brief    Compute the difference of ZTDDs.
 *  \param    f     ZTDD
 *  \param    g     ZTDD
 *  \return   The computed ZTDD
 *  \note     The output ZTDD accepts sets that are accepted by f and that are not accepted by g.
 */
struct mytdd *myztdd_diff(struct mytdd *f, struct mytdd *g)
{
  assert(f != NULL); assert(g != NULL);
  if(f == g || f == mytdd_bot)  return mytdd_bot;
  if(g == mytdd_bot)            return f;
  if(g == mytdd_top) {
    INC_RECDEPTH(recdepth);
    struct mytdd *r = myztdd_node(mytdd_itemval(f), myztdd_diff(f->ze, g), f->ne, f->po);
    ENSURE_TRUE_MSG(r != NULL, "ZTDD operation failed");
    DEC_RECDEPTH(recdepth);
    return r;
  }
  if(f == mytdd_top) {
    INC_RECDEPTH(recdepth);
    struct mytdd *r = myztdd_diff(f, g->ze);
    DEC_RECDEPTH(recdepth);
    return r;
  }

  struct mytdd *r;
  if(opcache_search(MYZTDD_DIFF_OP, (uintptr_t)f, (uintptr_t)g, (uintptr_t*)&r, mytdd_opcache)) return r;

  INC_RECDEPTH(recdepth);
  if(mytdd_itemval(f) > mytdd_itemval(g)) {
    r = myztdd_diff(f, g->ze);
  } else if (mytdd_itemval(f) < mytdd_itemval(g)) {
    r = myztdd_node(mytdd_itemval(f), 
                    myztdd_diff(f->ze, g), 
                    f->ne, 
                    f->po);
  } else {
    r = myztdd_node(mytdd_itemval(f), 
                    myztdd_diff(f->ze, g->ze), 
                    myztdd_diff(f->ne, g->ne), 
                    myztdd_diff(f->po, g->po));
  }
  ENSURE_TRUE_MSG(r != NULL, "ZTDD operation failed");
  DEC_RECDEPTH(recdepth);
  
  opcache_insert(MYZTDD_DIFF_OP, (uintptr_t)f, (uintptr_t)g, (uintptr_t)r, mytdd_opcache);
  return r;
}


/** \brief    Create an array and set all internal nodes in f so that they form an array of linked lists.
 *  \param    m     The maximum item value in f
 *  \param    f     TDD
 *  \see      mytdd_destory_arrayoflists
 *  \note     After execution, the aux fields of internal nodes are used as links.
 */
struct mytdd **mytdd_create_arrayoflists(itemval m, struct mytdd *f)
{
  assert(f != NULL);
#ifdef MY_DEBUG
  ENSURE_TRUE(mytdd_isconsistent(f));
#endif /*MY_DEBUG*/

  if(f == mytdd_top || f == mytdd_bot) {
    ENSURE_TRUE_WARN(false, "constant TDD was given");
    return NULL;
  }

  assert(m > 0);
  struct mytdd **a = (struct mytdd **)malloc(sizeof(struct mytdd*) * (m+1));
  ENSURE_TRUE_MSG(a != NULL, "memory allocation failed");
  mytdd_add_allocated_size(a);
  for(uintmax_t i = 0; i <= m; i++) a[i] = NULL;

  mytdd_make_arrayoflists(f, a);
  mytdd_unsetvisited_rec(f);
  return a;
}


void mytdd_destroy_arrayoflists(struct mytdd **a)
{
  assert(a != NULL);
  mytdd_del_allocated_size(a);
  free(a);
}

/** \brief    Make an array of linked lists so that each list consists of internal TDD nodes with the same item value.
 *  \param    f     TDD
 *  \param    a     Array of linked lists
 *  \note     After execution, the aux fields of internal nodes are used as links and the visited fields are set.
 */
static void mytdd_make_arrayoflists(struct mytdd *f, struct mytdd **a)
{
  assert(f != NULL); assert(a != NULL);
  if(mytdd_isvisited(f)) return;

  INC_RECDEPTH(recdepth);
  mytdd_make_arrayoflists(f->ze, a);
  mytdd_make_arrayoflists(f->ne, a);
  mytdd_make_arrayoflists(f->po, a);
  DEC_RECDEPTH(recdepth);

  mytdd_setaux((uintptr_t)a[mytdd_itemval(f)], f);
  a[mytdd_itemval(f)] = f;
  mytdd_setvisited(f);
}

/** \brief    Compute the intersection of DTDDs.
 *  \param    f     DTDD
 *  \param    g     DTDD
 *  \return   The computed DTDD
 *  \note     The output DTDD accepts sets that are accepted by both f and g.
 */
struct mytdd *mydtdd_intersect(struct mytdd *f, struct mytdd *g)
{
  assert(f != NULL); assert(g != NULL);
  if(f == mytdd_bot || g == mytdd_bot)  return mytdd_bot;
  if(f == mytdd_top)                    return g;
  if(g == mytdd_top)                    return f;
  if(f == g)                            return f;

  struct mytdd *r;
  if(opcache_search(MYDTDD_INT_OP, (uintptr_t)f, (uintptr_t)g, (uintptr_t*)&r, mytdd_opcache)) return r;

  struct mytdd *r0; struct mytdd *r1; struct mytdd *r2;
  INC_RECDEPTH(recdepth);
  if(mytdd_itemval(f) > mytdd_itemval(g)) {
    r0 = mydtdd_intersect(f,     g->ze);
    r1 = mydtdd_intersect(f,     g->ne);
    r2 = mydtdd_intersect(f,     g->po);
    r  = mydtdd_node(mytdd_itemval(g), r0, r1, r2);
  } else if (mytdd_itemval(f) < mytdd_itemval(g)) {
    r0 = mydtdd_intersect(f->ze, g);
    r1 = mydtdd_intersect(f->ne, g);
    r2 = mydtdd_intersect(f->po, g);
    r  = mydtdd_node(mytdd_itemval(f), r0, r1, r2);
  } else {
    r0 = mydtdd_intersect(f->ze, g->ze);
    r1 = mydtdd_intersect(f->ne, g->ne);
    r2 = mydtdd_intersect(f->po, g->po);
    r  = mydtdd_node(mytdd_itemval(g), r0, r1, r2);
  }
  DEC_RECDEPTH(recdepth);
  ENSURE_TRUE_MSG(r != NULL, "TDD operation failed");
  
  opcache_insert(MYDTDD_INT_OP, (uintptr_t)f, (uintptr_t)g, (uintptr_t)r, mytdd_opcache);
  return r;
}


/** \brief  Enlarge a uniquetable.
 *  \see    mytdd_node
 */
void mytdd_ut_enlarge(void)
{
  const uintmax_t oldsize = mytdd_ut_bucket_count;
  ENSURE_TRUE_MSG(oldsize < (UINTMAX_MAX >> MYTDD_UT_ENLARGEMENT_WIDTH), "ut size overflow");
  const uintmax_t newsize = oldsize << MYTDD_UT_ENLARGEMENT_WIDTH;
#ifdef UT_LOG 
  printf("ut_resizing\t%ju\t%ju\n", oldsize, newsize);
#endif /*UT_LOG*/
  struct mytdd **oldut = mytdd_ut;
  struct mytdd **newut = (struct mytdd**) malloc(sizeof(struct mytdd*) * newsize);
  ENSURE_TRUE_MSG(newut != NULL, "memory allocation failed");
  mytdd_add_allocated_size(newut);
  for(uintmax_t i = 0; i < newsize; i++) newut[i] = NULL;
#ifdef UT_LOG
  uintmax_t maxdepth    = 0;
#endif /*UT_LOG*/
  for(uintmax_t i = 0; i < oldsize; i++) {
    uintmax_t depth = 0;
    for(struct mytdd *c = oldut[i]; c != NULL;) {
      struct mytdd    *t  = c->nx;
      const uintmax_t val = mytdd_hash(mytdd_itemval(c), c->ze, c->ne, c->po, newsize);
      c->nx       = newut[val];
      newut[val]  = c;
      c           = t;
      depth++;
    }
#ifdef UT_LOG
  if(maxdepth < depth)  maxdepth = depth;
#endif /*UT_LOG*/
  }
  mytdd_del_allocated_size(oldut);
  free(oldut); oldut = NULL;
  mytdd_ut                  = newut;
  mytdd_ut_bucket_count     = newsize;
  mytdd_ut_node_count_threshold  = mytdd_ut_bucket_count * MYTDD_UT_MAX_LOADFACTOR;

#ifdef UT_LOG
  printf("ut_max_collision\t%ju\n", maxdepth);
  mytdd_printstat();
  mytdd_call_count      = 0;
  mytdd_collision_count = 0;
#endif /*UT_LOG*/

  opcache_enlarge(mytdd_opcache);
  mytdd_opcache_allocated_size = opcache_allocsize(mytdd_opcache);

#ifdef UT_LOG
  printf("\n");
#endif /*UT_LOG*/

}

/** \brief  Print the current status of uniquetable and operation cache to stdout.
 *  \note   To enable this function, define the macro UT_LOG in Makefile.
 */
void mytdd_printstat(void)
{
#ifdef UT_LOG
  if(mytdd_call_count != 0) printf("ut_average_collision\t%.2f\n", (double)mytdd_collision_count/mytdd_call_count);
  opcache_printstat(mytdd_opcache);
  printf("ut_mem_usage\t%zu\n", mytdd_ut_allocated_size + mytdd_opcache_allocated_size);
#endif /*UT_LOG*/
}

/** \breif    Make a dot file for a TDD.
 *  \param    m     The maximum item value in f
 *  \param    f     TDD
 *  \param    out   Pointer to a created dot file
 *  \return   ST_SUCCESS if successful; otherwise, ST_FAILURE.
 *  \note     Use Graphviz to visualize a dot file.
 */
int mytdd_to_dot(itemval m, struct mytdd *f, FILE *out)
{ 
  assert(f != NULL); assert(out != NULL);
#ifdef MY_DEBUG
  ENSURE_TRUE(mytdd_isconsistent(f));
#endif /*MY_DEBUG*/

  if(f == mytdd_top || f == mytdd_bot) {
    ENSURE_TRUE_WARN(false, "invalid input");
    return ST_FAILURE;
  }

  int res = fseek(out, 0L, SEEK_SET);
  ENSURE_SUCCESS(res);

  struct mytdd **a = mytdd_create_arrayoflists(m, f);
  uintmax_t index = 2; // bottom node has 0, and top node has 1 as their indices.
  mytdd_setaux(0, mytdd_bot);
  mytdd_setaux(1, mytdd_top);
  fprintf(out, "digraph tdd {\n");
  fprintf(out, "{rank = same; 0 1}\n");
  for(uintmax_t i = 1; i <= m; i++) {
    fprintf(out, "{rank = same;");
    for(struct mytdd *n = a[i]; n != NULL; n = (struct mytdd*)n->aux) fprintf(out, " %ju", index++);
    fprintf(out, "}\n");
  }
  for(uintmax_t i = m; i > 0; i--) {
  for(struct mytdd *n = a[i]; n != NULL;) {
      struct mytdd *t = (struct mytdd*)n->aux;
      index--;
      fprintf(out, "%ju [label = %d];\n",           index, mytdd_itemval(n));
      fprintf(out, "%ju -> %ju ;\n",                 index, (uintmax_t)n->po->aux);
      fprintf(out, "%ju -> %ju [style = dashed];\n", index, (uintmax_t)n->ne->aux);
      fprintf(out, "%ju -> %ju [style = dotted];\n", index, (uintmax_t)n->ze->aux);
      mytdd_setaux(index, n);
      n = t;
    }
  }
  assert(index == 2);
  fprintf(out, "1 [shape=box];\n");
  fprintf(out, "0 [shape=box];\n");
  fprintf(out, "}\n");

  mytdd_destroy_arrayoflists(a);
  return ST_SUCCESS;
}



