/** \file     my_io_dimacs.h
 *  \brief    IO functions for data files in DIMACS CNF format
 *  \author   Takahisa Toda
 *  \note     See http://www.cs.ubc.ca/~hoos/SATLIB/benchm.html for benchmark problems and details of DIMACS CNF format.
 */
#ifndef MY_IO_DIMACS_H
#define MY_IO_DIMACS_H

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <math.h>
#include "my_def.h"

#define CH_DIMACS_COMMENT   'c' //!< comment line character of DIMACS format
#define CH_DIMACS_PROBLEM   'p' //!< problem line character of DIMACS format
#define MAX_LINELEN         (UINTMAX_C(1) << 20)  //!< the maximum length of a single line in a datafile

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

/** \brief  An explicite data represention of a set family
 *  \note
 *  - Let sf point to a setfam data structure.
 *  - sf->elem[i-1][j-1] corresponds to the j-th element in the i-th set. 
 *  - Each set ends with INFTY, that is,
 *  - sf->elem[i-1][k] == INFTY if the i-th set contains exactly k elements.
 */
struct setfam {
  uintmax_t card;      //!< the number of sets with repetition, i.e. cardinality
  uintmax_t memlen;    //!< the length of mem
  itemval   **elem;    //!< double pointer to an element in mem
  itemval   *mem;      //!< memory area (array) of elements, which are partitioned into sets using INFTY as delimiter.
};

/*---------------------------------------------------------------------------*/
/* Function prototypes                                                       */
/*---------------------------------------------------------------------------*/
extern int              getfileinfo_dimacs          (uintmax_t *pn, itemval *pm, uintmax_t *pt, FILE *in);
extern struct setfam    *create_setfam_from_dimacs  (uintmax_t n, uintmax_t t, FILE *in);
extern void             destroy_setfam              (struct setfam *sf);


/** \brief Compute a dynamically allocated memory size in byte.
 *  \param h      Pointer to a setfam data structure
 *  \return The computed size in byte
 */
static inline uintmax_t setfam_allocsize            (struct setfam *sf)
{
  return malloc_usable_size(sf) + malloc_usable_size(sf->elem) + malloc_usable_size(sf->mem); 
}

/** \brief    Compare item values with respect to their absolute values.
 *  \param    a     Pointer to an item value
 *  \param    b     Pointer to an item value
 *  \return   1 if a is greater; -1 if b is greater; 0 otherwise.
 */
static inline int comp_itemval(const void *a, const void *b)
{
  const itemval c = *((const itemval*)a);
  const itemval d = *((const itemval*)b);
  
  if (abs(c) > abs(d))
    return 1;
  else if (abs(c) < abs(d))
    return -1;
  else
    return 0;
}

/** \brief    Sort each set in increasing order and remove duplicated elements.
 *  \param    x     Double pointer to an element 
 *  \param    n     The number of sets with repetition
 */
static inline void format_eachrow(itemval *x[], uintmax_t n)
{
  for(uintmax_t i = 0; i < n; i++) {
    uintmax_t d = 0; while(x[i][d] != INFTY) {d++;}
    qsort(x[i], d, sizeof(itemval), comp_itemval);
    itemval prev = 0;
    for(uintmax_t j = 0; j < d; j++) {
      ENSURE_TRUE_MSG(abs(prev) != abs(x[i][j]) || prev != x[i][j], "duplicated entries found");
      prev = x[i][j];
    }
  }
}


#endif /*MY_IO_DIMACS_H*/
