/** \file     my_io_dimacs.c
 *  \brief    Implementation of IO functions for data files in DIMACS cnf format
 *  \author   Takahisa Toda
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <assert.h>

#include "my_def.h"
#include "my_io_dimacs.h"

/** \brief    Check if a character is a delimiter of DIMACS format.
 *  \param    c     Character
 *  \return   true if yes; otherwise, false.
 */
static bool isdelimiter(char c)
{
  return c == ' ' || c == '\n' || c == '\t';
}


/** \brief    Obtain information from a data file in DIMACS format.
 *  \param    pclause_count   Pointer to a variable, in which the number of clauses with repetition is written.
 *  \param    pmaxval         Pointer to a variable, in which the maximum item value (variable index) is written.
 *  \param    pliteral_count  Pointer to the number of literals with repetition
 *  \param    in              Pointer to a dafa file
 *  \return   ST_SUCCESS if successful; otherwise, ST_FAILURE.
 */
int getfileinfo_dimacs(uintmax_t *pclause_count, itemval *pmaxval, uintmax_t *pliteral_count, FILE *in)
{
  assert(pclause_count != NULL); assert(pmaxval != NULL); assert(pliteral_count != NULL); assert(in != NULL);
  int res = fseek(in, 0L, SEEK_SET);
  ENSURE_SUCCESS(res);
  char buf[MAX_LINELEN];
  bool isfound = false; // Is the 1st literal of the currently reading clause found?

  *pclause_count = *pmaxval = *pliteral_count = 0;
  while(fgets(buf, MAX_LINELEN, in) != NULL) {
    if(buf[0] == CH_DIMACS_COMMENT) continue;
    if(buf[0] == CH_DIMACS_PROBLEM) {
#ifdef MISC_LOG
      printf("problem\t"); printf("%s", buf); 
#endif /*MISC_LOG*/
      continue;
    }
    for(uintmax_t s = 0, t = 0; buf[s] != '\0'; s = t) {
      while(isdelimiter(buf[s])  && buf[s] != '\0') {s++; assert(s < MAX_LINELEN);} // Find the first non-delimiter.
      t = s;
      while(!isdelimiter(buf[t]) && buf[t] != '\0') {t++; assert(t < MAX_LINELEN);} // Find the first delimiter.
      if(isdelimiter(buf[t])) {buf[t++] = '\0'; assert(t < MAX_LINELEN);}
      if(s != t) {
        itemval v = atoi(buf + s);
        if(v == 0) {ENSURE_TRUE_MSG(isfound, "empty clause found"); isfound = false;} // finished reading the current clause.
        else       {(*pliteral_count)++; if(!isfound) {isfound = true; (*pclause_count)++;}}
        if(*pmaxval < abs(v)) *pmaxval = abs(v);
      }
    }
  }

#ifdef MISC_LOG
  printf("maxval\t%d\n",     *pmaxval);
  printf("#clauses\t%ju\n",  *pclause_count);
  printf("#literals\t%ju\n", *pliteral_count);
  printf("\n");
#endif /*MISC_LOG*/
  return ST_SUCCESS;
}


/** \brief    Create a setfam data structure and set a DIMACS CNF data to it.
 *  \param    clause_count      The number of clauses with repetition
 *  \param    literal_count     The number of clauses with repetition
 *  \param    in                Pointer to a DIMACS CNF dafafile
 *  \return   The created setfam data structure
 */
struct setfam *create_setfam_from_dimacs(uintmax_t clause_count, uintmax_t literal_count, FILE *in)
{
  assert(in != NULL);
  int res = fseek(in, 0L, SEEK_SET);
  ENSURE_SUCCESS(res);

  const uintmax_t len = clause_count + literal_count;
  struct setfam *sf   = (struct setfam*)malloc(sizeof(struct setfam));
  ENSURE_TRUE_MSG(sf != NULL, "memory allocation failed");
  sf->elem    = (itemval**)malloc(sizeof(itemval*) * clause_count);
  ENSURE_TRUE_MSG(sf->elem != NULL, "memory allocation failed");
  sf->mem     = (itemval*)malloc(sizeof(itemval) * len);
  ENSURE_TRUE_MSG(sf->mem  != NULL, "memory allocation failed");
  sf->card    = clause_count;
  sf->memlen  = len;

  itemval **set = sf->elem;
  itemval *m    = sf->mem;

  char buf[MAX_LINELEN];
  bool isfound = false; // Is the 1st literal of the currently reading clause found?

  while(fgets(buf, MAX_LINELEN, in) != NULL) {
    if(buf[0] == CH_DIMACS_COMMENT || buf[0] == CH_DIMACS_PROBLEM) continue;
    for(uintmax_t s = 0, t = 0; buf[s] != '\0'; s = t) {
      while(isdelimiter(buf[s])  && buf[s] != '\0') {s++; assert(s < MAX_LINELEN);} // Find the first non-delimiter.
      t = s;
      while(!isdelimiter(buf[t]) && buf[t] != '\0') {t++; assert(t < MAX_LINELEN);} // Find the first delimiter.
      if(isdelimiter(buf[t])) {buf[t++] = '\0'; assert(t < MAX_LINELEN);}
      if(s != t) {
        itemval v = atoi(buf + s);
        assert(m < sf->mem + len);
        if(v == 0) {
          isfound = false; // finished reading the current clause.
          *m = INFTY; m++;
        } else {
          if(!isfound) {isfound = true; assert(set < sf->elem + clause_count); *set = m; set++;}
          *m = v; m++;
        }
      }
    }
  }
  assert(m    == sf->mem  + len);
  assert(set  == sf->elem + clause_count);

#ifdef MISC_LOG
  printf("dimacs_to_setfam\t%zu\tbyte\n", setfam_allocsize(sf));  
#endif /*MISC_LOG*/

  return sf;
}


/** \brief   Free used memory area.
 *  \param   sf     Pointer to setfam
 */
void destroy_setfam(struct setfam *sf)
{
  assert(sf != NULL);
  free(sf->elem); free(sf->mem); free(sf);
}



