/** \file   dbtdd.c
 *  \author Takahisa Toda
 *  \see    T.Toda, Dualization of Boolean functions Using Ternary Decision Diagrams, in Proc. of 13th International Symposium on Artificial Intelligence and Mathematics (ISAIM2014), Florida, USA(2014).
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include "my_def.h"
#include "rng.h"
#include "my_opcache.h"
#include "my_io_dimacs.h"
#include "tdd_io_dimacs.h"
#include "tdd_gen.h"
#include "tdd_min.h"

#define PROGNAME "dbtdd"
#define PROGVERS "2.0.0"

#ifdef USE_BDD
#include "bdd_interface.h"
#include "dnf2bdd.h"
#include "bdd_gen.h"
#include "bdd_min.h"
#endif

#ifdef SELECT_CUDD
#ifndef USE_BDD
#error "If SELECT_CUDD is defined, USE_BDD must be defined as well."
#endif 
DdManager *dd_mgr = NULL; //!< BDD/ZDD manager for CUDD
#endif /*SELECT_CUDD*/

#ifdef SELECT_GSL
my_rng *g_rng; //!< a random number generator provided by GSL
#endif /*SELECT_GSL*/


#define PRINT_USAGE(p) do{fprintf(stderr, "Usage:\t%s input-file [output-file]\n", (p));                                            \
    fprintf(stderr, "%s\tVersion %s\n", PROGNAME, PROGVERS);                                                                        \
    fprintf(stderr, "Description: given a DNF of a Boolean functions in DIMACS CNF, %s computes the complete DNF of the dual function.\n", PROGNAME);    \
    fprintf(stderr, "-p\tCompute all prime implicants from implicants.\n");                                                         \
    fprintf(stderr, "-m<int>\tMaximum usable memory for TDD uniquetable in Mbyte: place integer without space after 'm'.\n");       \
  } while(0)

#if defined(MISC_LOG) || defined(SIZE_LOG)
  struct tm *execdate;
#endif /*defined(MISC_LOG) || defined(SIZE_LOG)*/


int main(int argc, char *argv[])
{
    int res       = ST_SUCCESS;
    char *infile  = NULL;
    char *outfile = NULL;
    int minonly   = false;
    uintmax_t maxsize = 0;
  
  
    /*** RECEIVE INPUTS ***/  
    for (int i = 1; i < argc; i++) {
        if (argv[i][0] == '-') {
            switch (argv[i][1]){
                case 'm':
                    maxsize = (uintmax_t)strtoul(argv[i]+2, NULL, 0);
                    maxsize = maxsize << 20; // convert into byte.
                    break;
                case 'p':
                    minonly = true; 
                    break;
                case 'v': case '?': case 'h': default:
                    PRINT_USAGE(argv[0]); return  ST_SUCCESS;
            }
        } else {
        if (infile == NULL) 
            {infile  = argv[i];}
        else if (outfile == NULL)
            {outfile = argv[i];}
        else
            {PRINT_USAGE(argv[0]); return  ST_SUCCESS;}
        }
    }
    if (infile == NULL)
        {PRINT_USAGE(argv[0]); return  ST_SUCCESS;}
#if defined(MISC_LOG) || defined(SIZE_LOG)
    time_t timer = time(NULL);
    execdate     = localtime(&timer);
#endif /*defined(MISC_LOG) || defined(SIZE_LOG)*/
#ifdef MISC_LOG
    printf("date\t%s",          asctime(execdate));
    printf("program\t%s-%s\n",  PROGNAME, PROGVERS);
    printf("package\t%s\n",     TDD_PACKAGE);
    printf("input\t%s\n",       infile);
    if (outfile != NULL)
        printf("output\t%s\n", outfile);
    printf("max ut size(MB)\t%ju\n",  maxsize >> 20);
    printf("\n");
#endif /*MISC_LOG*/

    FILE *in = fopen(infile, "r");
    ENSURE_TRUE_MSG(in != NULL, "file open failed\n");
  
    /*** SETUP TDD PACKAGE ***/
    rng_create();
    uintmax_t clause_count, literal_count;
    itemval maxval;
    res = getfileinfo_dimacs(&clause_count, &maxval, &literal_count, in);
    ENSURE_SUCCESS(res);
    ENSURE_TRUE_MSG(maxval <= TDD_MAXITEMVAL, "the number of variable too large");
    tdd_init(0,0,0,maxsize);
 
#ifdef USE_BDD
    bdd_init(maxval, 0);
#endif

    clock_t startingtime = current_time();
    ztddp r;
    if (!minonly) {
        /*** CONSTRUCT ZTDD FROM DIMACS CNF DATAFILE ***/
        ztddp f = ztdd_from_dimacs(clause_count, literal_count, in);
        ENSURE_TRUE_MSG(f != TDD_NULL, "ZTDD construction failed");
        print_elapsed_time(startingtime, "INPUT");
#ifdef MISC_LOG
        printf("|ztdd|\t%ju\n",     ztdd_size(f));
        printf("#sets\t%ju\n",      ztdd_card(f));
        printf("uncmp_size\t%ju\n", ztdd_uncmpsize(f));
        printf("\n");
        fflush(stdout);
#endif /*MISC_LOG*/

#ifdef USE_BDD
        /*** CONSTRUCT BDD FOR ALL TOTAL SATISFYING ASSIGNMENTS ***/
        startingtime = current_time();
        bddp g = gen_z2b(f);
        ENSURE_TRUE(g != BDD_NULL);
        print_elapsed_time(startingtime, "GEN");
#ifdef MISC_LOG
        printf("|bdd|\t%ju\n",         bdd_size(g));
        printf("\n");
        fflush(stdout);
#endif /*MISC_LOG*/

        /*** MINIMIZATION BY RESTRICTING TO PRIME IMPLICANTS ***/
        startingtime = current_time();
        r = min_b2z(g);
        ENSURE_TRUE(r != TDD_NULL);
        print_elapsed_time(startingtime, "MIN");
#ifdef MISC_LOG
        printf("|ztdd|\t%ju\n",     ztdd_size(r));
        printf("#sets\t%ju\n",      ztdd_card(r));
        printf("uncmp_size\t%ju\n", ztdd_uncmpsize(r));
        printf("\n");
        fflush(stdout);
#endif /*MISC_LOG*/

#else /*USE_DTDD*/
        /*** GENERATE ALL IMPLICANTS ***/
        startingtime = current_time();
        dtddp g = gen_z2d(f);
        ENSURE_TRUE(g != TDD_NULL);
        print_elapsed_time(startingtime, "GEN");
#ifdef MISC_LOG
        printf("|dtdd|\t%ju\n",         dtdd_size(g));
        printf("\n");
        //printf("#sets\t%ju\n",        dtdd_card(maxval, g));
        //printf("uncmp_size\t%ju\n\n", dtdd_uncmpsize(maxval, g));
        fflush(stdout);
#endif /*MISC_LOG*/

        /*** MINIMIZATION BY RESTRICTING TO PRIME IMPLICANTS ***/
        startingtime = current_time();
        r = min_d2z(g);
        ENSURE_TRUE(r != TDD_NULL);
        print_elapsed_time(startingtime, "MIN");
#ifdef MISC_LOG
        printf("|ztdd|\t%ju\n",     ztdd_size(r));
        printf("#sets\t%ju\n",      ztdd_card(r));
        printf("uncmp_size\t%ju\n", ztdd_uncmpsize(r));
        printf("\n");
        fflush(stdout);
#endif /*MISC_LOG*/
#endif

    } else {
#ifdef USE_BDD
        /*** CONVERTING DTDD to BDD ****/
        struct setfam *sf = create_setfam_from_dimacs(clause_count, literal_count, in);
        ENSURE_TRUE(sf != NULL);

        startingtime = current_time();
        bddp f = dnf2bdd(sf);
        ENSURE_TRUE_MSG(f != BDD_NULL, "BDD operation failed");
        print_elapsed_time(startingtime, "CONV");
#ifdef MISC_LOG
        printf("|bdd|\t%ju\n",      bdd_size(f));
        printf("\n");
        fflush(stdout);
#endif /*MISC_LOG*/

        /*** MINIMIZATION BY RESTRICTING TO PRIME IMPLICANTS ***/
        startingtime = current_time();
        r = min_b2z(f);
        ENSURE_TRUE(r != TDD_NULL);
        print_elapsed_time(startingtime, "MIN");
#ifdef MISC_LOG
        printf("|ztdd|\t%ju\n",     ztdd_size(r));
        printf("#sets\t%ju\n",      ztdd_card(r));
        printf("uncmp_size\t%ju\n", ztdd_uncmpsize(r));
        printf("\n");
        fflush(stdout);
#endif /*MISC_LOG*/
#else
        fprintf(stderr, "error:%s:%d:%s: set BDD to DDTYPE in Makefile to use p option.\n", __FILE__, __LINE__, __func__);
        exit(EXIT_FAILURE);
#endif
    }


    /*** OUTPUT ZTDD ***/
    if (outfile != NULL) {
        FILE *out = fopen(outfile, "w");
        ENSURE_TRUE_MSG(out != NULL, "file open failed\n");
        startingtime = current_time();
        ztdd_to_dimacs(maxval, r, out, outfile);
        //ztdd_to_dot(maxval, r, out);
        print_elapsed_time(startingtime, "OUTPUT");
        fclose(out);
    }

#ifdef USE_BDD
    bdd_quit();
#endif

    tdd_printstat();
    tdd_quit();
    rng_destroy();
    fclose(in);

    return ST_SUCCESS;
}
