/** \file     bdd_gen.c
 *  \brief    Generation of All Total Satisfying Assignments
 *  \author   Takahisa Toda
 *  \see      T.Toda, Dualization of Boolean functions Using Ternary Decision Diagrams, in Proc. of 13th International Symposium on Artificial Intelligence and Mathematics (ISAIM2014), Florida, USA(2014).
 */

#include "bdd_gen.h"
#include "my_hash.h"

static bddp gen_z2b_rec(ztddp f, my_hash *h);

static uintmax_t recdepth = 0; //!< recursion depth

#ifdef SIZE_LOG
static FILE       *sizlog = NULL;
static uintmax_t  maxsize = 0; //!< the maximum size of intermediate BDDs
#endif /*SIZE_LOG*/


/** \brief    Construct the BDD for all total satisfying assignments from a ZTDD, where a ZTDD is considered as a representation of CNFs.
 *  \param    f     ZTDD
 *  \return   The BDD contructed.
 *  \note     To log output BDD sizes, define the macro SIZE_LOG in Makefile. The result is outputed as a file, where the 1st column consists of input sizes and the 2nd column output sizes.
 */
bddp gen_z2b(ztddp f)
{
  my_hash *h = ht_create(0);
  ENSURE_TRUE_MSG(h != NULL, "hash table creation failed");

#ifdef SIZE_LOG
  char str[BUFSIZ];
  strftime(str, BUFSIZ, "bdd_gen%Y%m%d%H%M.log", execdate);
  sizlog = fopen(str, "w");
  ENSURE_TRUE_MSG(sizlog != NULL, "file open failed\n");
  maxsize = 0;
#endif /*SIZE_LOG*/

  assert(recdepth == 0);
  bddp r = gen_z2b_rec(f, h);
  ENSURE_TRUE(r != BDD_NULL);
  assert(recdepth == 0);

#ifdef SIZE_LOG
  fclose(sizlog);
  printf("max|bdd|\t%ju\n", maxsize);
#endif /*SIZE_LOG*/

  ht_destroy(h);
  return r;
}

static bddp gen_z2b_rec(ztddp f, my_hash *h)
{
  if(f == ztdd_top()) return bdd_bot();
  if(f == ztdd_bot()) return bdd_top();

  bddp r;
  if(ht_search((uintptr_t)f, (uintptr_t*)&r, h)) return r;

  bddp t0 = gen_z2b_rec(ztdd_zero(f), h);  
  bddp t1 = gen_z2b_rec(ztdd_neg(f),  h);
  bddp t2 = gen_z2b_rec(ztdd_pos(f),  h);

  INC_RECDEPTH(recdepth);
  bddp hi = bdd_and(t0, t1); 
  ENSURE_TRUE_MSG(hi != TDD_NULL, "BDD operation failed");
  bddp lo = bdd_and(t0, t2); 
  ENSURE_TRUE_MSG(lo != TDD_NULL, "BDD operation failed");
  DEC_RECDEPTH(recdepth);

  r = bdd_node(ztdd_itemval(f), lo, hi);
  ENSURE_TRUE_MSG(r != TDD_NULL,  "BDD operation failed");

  ht_insert((uintptr_t)f, (uintptr_t)r, h);
  
#ifdef SIZE_LOG
  const uintmax_t insize  = ztdd_size(f);
  const uintmax_t outsize = bdd_size(r);
  fprintf(sizlog,"%ju\t%ju\n", insize, outsize);
  if(maxsize < outsize) maxsize = outsize;
#endif /*SIZE_LOG*/

  return r;
}

